<?php

namespace App\Http\Controllers;

use App\Models\Task;
use App\Support\XlsxExporter;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;

class TaskController extends Controller
{
    public function index(Request $request)
    {
        $query = Task::query();

        if ($request->filled('status')) {
            $query->where('status', $request->input('status'));
        }

        if ($request->filled('type')) {
            $query->whereJsonContains('types', $request->input('type'));
        }

        if ($request->filled('month')) {
            $month = $request->input('month');
            $query->whereRaw("DATE_FORMAT(COALESCE(recorded_at, created_at), '%Y-%m') = ?", [$month]);
        }

        if ($request->filled('q')) {
            $q = trim($request->input('q'));
            $query->where(function ($qb) use ($q) {
                $qb->where('name', 'like', "%{$q}%")
                   ->orWhere('reminder_email', 'like', "%{$q}%")
                   ->orWhereJsonContains('reminder_emails', $q)
                   ->orWhere('status', 'like', "%{$q}%");
                $code = strtoupper($q);
                if (in_array($code, ['SA','DB','FE','BE','SI','AS'])) {
                    $qb->orWhereJsonContains('types', $code);
                }
            });
        }

        $tasks = $query->orderByDesc('created_at')->paginate(10)->appends($request->query());

        return view('tasks.index', compact('tasks'));
    }

    public function create()
    {
        $this->authorize('manage-tasks');
        return view('tasks.create');
    }

    public function store(Request $request)
    {
        $this->authorize('manage-tasks');
        $data = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'link_gitlab' => ['nullable', 'url'],
            'target_date' => ['required', 'date'],
            'types' => ['nullable', 'array'],
            'types.*' => ['in:SA,DB,FE,BE,SI,AS'],
            'reminder_frequency_hours' => ['required', 'in:4,8,12,24'],
            'reminder_emails' => ['required', 'array', 'min:1'],
            'reminder_emails.*' => ['email'],
        ]);

        $data['recorded_at'] = Carbon::now()->toDateString();
        $data['status'] = 'Tersampaikan';
        $data['reminder_email'] = $data['reminder_emails'][0] ?? '';

        Task::create($data);

        return redirect()->route('tasks.index')->with('success', 'Task ditambahkan.');
    }

    public function show(Task $task)
    {
        return view('tasks.show', compact('task'));
    }

    public function edit(Task $task)
    {
        $this->authorize('manage-tasks');
        return view('tasks.edit', compact('task'));
    }

    public function update(Request $request, Task $task)
    {
        $this->authorize('manage-tasks');
        $data = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'link_gitlab' => ['nullable', 'url'],
            'target_date' => ['required', 'date'],
            'types' => ['nullable', 'array'],
            'types.*' => ['in:SA,DB,FE,BE,SI,AS'],
            'status' => ['required', 'in:Tersampaikan,Sedang Dikerjakan,Selesai'],
            'reminder_frequency_hours' => ['required', 'in:4,8,12,24'],
            'reminder_emails' => ['required', 'array', 'min:1'],
            'reminder_emails.*' => ['email'],
        ]);

        $data['reminder_email'] = $data['reminder_emails'][0] ?? $task->reminder_email;
        $task->update($data);

        return redirect()->route('tasks.index')->with('success', 'Task diperbarui.');
    }

    public function destroy(Task $task)
    {
        $task->delete();
        return redirect()->route('tasks.index')->with('success', 'Task dihapus.');
    }

    public function export(Request $request)
    {
        $query = Task::query();
        if ($request->filled('status')) {
            $query->where('status', $request->input('status'));
        }
        if ($request->filled('type')) {
            $query->whereJsonContains('types', $request->input('type'));
        }
        if ($request->filled('month')) {
            $month = $request->input('month');
            $query->whereRaw("DATE_FORMAT(COALESCE(recorded_at, created_at), '%Y-%m') = ?", [$month]);
        }
        if ($request->filled('q')) {
            $q = trim($request->input('q'));
            $query->where(function ($qb) use ($q) {
                $qb->where('name', 'like', "%{$q}%")
                   ->orWhere('reminder_email', 'like', "%{$q}%")
                   ->orWhereJsonContains('reminder_emails', $q)
                   ->orWhere('status', 'like', "%{$q}%");
                $code = strtoupper($q);
                if (in_array($code, ['SA','DB','FE','BE','SI','AS'])) {
                    $qb->orWhereJsonContains('types', $code);
                }
            });
        }

        $headers = ['Nama Task', 'Link Gitlab', 'Jenis Task', 'Bulan'];
        $rows = [];
        foreach ($query->orderByDesc('created_at')->cursor() as $task) {
            $rows[] = [
                $task->name,
                $task->link_gitlab ?? '',
                implode(', ', $task->types ?? []),
                \Illuminate\Support\Carbon::parse($task->created_at)->translatedFormat('F, Y'),
            ];
        }

        $path = XlsxExporter::create($headers, $rows);
        $filename = 'tasks-export-' . now()->format('Ymd-His') . '.xlsx';
        return response()->download($path, $filename, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ])->deleteFileAfterSend(true);
    }
}
